/*
 * This file is part of DisOrder.
 * Copyright (C) 2008, 2009 Richard Kettlewell
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "test.h"
#include "cgi.h"

static void input_from(const char *s) {
  FILE *fp = tmpfile();
  char buffer[64];

  if(fputs(s, fp) < 0
     || fputs("wibble wibble\r\nspong", fp) < 0 /* ensure CONTENT_LENGTH
                                                 * honored */
     || fflush(fp) < 0)
    disorder_fatal(errno, "writing to temporary file");
  rewind(fp);
  xdup2(fileno(fp), 0);
  lseek(0, 0/*offset*/, SEEK_SET);
  snprintf(buffer, sizeof buffer, "%zu", strlen(s));
  setenv("CONTENT_LENGTH", buffer, 1);
}

static void test_cgi(void) {

  setenv("REQUEST_METHOD", "GET", 1);
  setenv("QUERY_STRING", "foo=bar&a=b+c&c=x%7ey", 1);
  cgi_init();
  check_string(cgi_get("foo"), "bar");
  check_string(cgi_get("a"), "b c");
  check_string(cgi_get("c"), "x~y");

  setenv("REQUEST_METHOD", "POST", 1);
  setenv("CONTENT_TYPE", "application/x-www-form-urlencoded", 1);
  unsetenv("QUERY_STRING");
  input_from("foo=xbar&a=xb+c&c=xx%7ey");
  cgi_init();
  check_string(cgi_get("foo"), "xbar");
  check_string(cgi_get("a"), "xb c");
  check_string(cgi_get("c"), "xx~y");

  /* This input string generated by Firefox 2.0.0.14 */
  input_from("-----------------------------16128946562344073111198667379\r\n"
             "Content-Disposition: form-data; name=\"input1\"\r\n"
             "\r\n"
             "normal input\r\n"
             "-----------------------------16128946562344073111198667379\r\n"
             "Content-Disposition: form-data; name=\"input2\"\r\n"
             "\r\n"
             "with pound sign: \xC2\xA3\r\n"
             "-----------------------------16128946562344073111198667379\r\n"
             "Content-Disposition: form-data; name=\"input3\"\r\n"
             "\r\n"
             "hidden input\r\n"
             "-----------------------------16128946562344073111198667379\r\n"
             "Content-Disposition: form-data; name=\"submit\"\r\n"
             "\r\n"
             "Submit Query\r\n"
             "-----------------------------16128946562344073111198667379--\r\n");
  setenv("CONTENT_TYPE", "multipart/form-data; boundary=---------------------------16128946562344073111198667379", 1);
  unsetenv("QUERY_STRING");
  cgi_init();
  check_string(cgi_get("input1"), "normal input");
  check_string(cgi_get("input2"), "with pound sign: \xC2\xA3");
  check_string(cgi_get("input3"), "hidden input");
  check_string(cgi_get("submit"), "Submit Query");

  input_from("-----------------------------33919340914020259251659146591\r\n"
             "Content-Disposition: form-data; name=\"text\"\r\n"
             "\r\n"
             "Text with\r\n"
             "multiple lines\r\n"
             "and trailing spaces \r\n"
             "---and leading dashes\r\n"
             "and pound sign \xC2\xA3\r\n"
             "\r\n"
             "-----------------------------33919340914020259251659146591\r\n"
             "Content-Disposition: form-data; name=\"empty\"\r\n"
             "\r\n"
             "\r\n"
             "-----------------------------33919340914020259251659146591\r\n"
             "Content-Disposition: form-data; name=\"submit\"\r\n"
             "\r\n"
             "Submit Query\r\n"
             "-----------------------------33919340914020259251659146591--\r\n");
  setenv("CONTENT_TYPE", "multipart/form-data; boundary=---------------------------33919340914020259251659146591", 1);
  cgi_init();
  check_string(cgi_get("text"), ("Text with\r\n"
                                 "multiple lines\r\n"
                                 "and trailing spaces \r\n"
                                 "---and leading dashes\r\n"
                                 "and pound sign \xC2\xA3\r\n"
                                 ""));
  check_string(cgi_get("empty"), "");
  
  check_string(cgi_sgmlquote("foobar"), "foobar");
  check_string(cgi_sgmlquote("<wibble>"), "&#60;wibble&#62;");
  check_string(cgi_sgmlquote("\"&\""), "&#34;&#38;&#34;");
  check_string(cgi_sgmlquote("\xC2\xA3"), "&#163;");

  check_string(cgi_makeurl("http://example.com/", (char *)NULL),
	       "http://example.com/");
  check_string(cgi_makeurl("http://example.com/",
			   "foo", "bar",
			   "a", "b c",
			   "d", "f=g+h",
			   (char *)NULL),
	       "http://example.com/?foo=bar&a=b%20c&d=f%3dg%2bh");
  
}

TEST(cgi);

/*
Local Variables:
c-basic-offset:2
comment-column:40
fill-column:79
indent-tabs-mode:nil
End:
*/
